#!/usr/bin/env perl

# bowtie_wrapper.pl
# MacVector
#
# Created by B. Kevin Hardman on 05/19/11
# Copyright © 2011 MacVector, Inc. All rights reserved.

use POSIX qw(setsid);
setsid;
$SIG{INT} = \&process_signal_INT;

use Getopt::Long;

my @firstMates;
my @secondMates;
my $arguments;
my $basename;
my $depth;
my $index;
my @interleaved;
my $path;
my @references;
my @sequences;
my $test;
my $threads;
my $unaligned;
my $verbose;
my $workingDirectory = "/tmp/macvector/bowtie";
my $result = GetOptions(
	"1=s" => \@firstMates,
	"2=s" => \@secondMates,
	"arguments=s" => \$arguments,
	"basename=s" => \$basename,
	"depth=s" => \$depth,
	"index=s" => \$index,
	"interleaved=s" => \@interleaved,
	"path=s" => \$path,
	"reference=s" => \@references,
	"sequence=s" => \@sequences,
	"test" => \$test,
	"threads=i" => \$threads,
	"unaligned=s" => \$unaligned,
	"verbose" => \$verbose,
	"working_directory=s" => \$workingDirectory
);

my $indexArguments;
$indexArguments = " -c" if $index =~ /^csi$/i;

# build reference argument and create reference sequence indices
my $referenceArgument;
foreach my $reference (@references) {
	$referenceArgument .= "$reference,";

	print STDERR qq{"$path/samtools" faidx "$reference"\n} if $verbose;
	print STDOUT qx{"$path/samtools" faidx "$reference"} if !$test;
}
chop $referenceArgument;

# build read argument
my $sequenceArgument;
if (@sequences) {
	$sequenceArgument .= "-U ";
	foreach my $sequence (@sequences) {
		$sequenceArgument .= qq{"$sequence",};
	}
	chop $sequenceArgument;
}

my $matesArgument;
if (@firstMates && @secondMates) {
	$matesArgument .= "-1 ";
	foreach my $firstMate (@firstMates) {
		$matesArgument .= qq{"$firstMate",};
	}
	chop $matesArgument;
	$matesArgument .= " -2 ";
	foreach my $secondMate (@secondMates) {
		$matesArgument .= qq{"$secondMate",}
	}
	chop $matesArgument;
	$matesArgument .= " " if $matesArgument;
}

my $interleavedArgument;
if (@interleaved) {
	$interleavedArgument .= "--interleaved ";
	foreach my $sequence (@interleaved) {
		$interleavedArgument .= qq{"$sequence",};
	}
	chop $interleavedArgument;
	$interleavedArgument .= " " if $interleavedArgument;
}

my $unalignedArgument;
if ($unaligned) {
	if (@sequences) {
		my $option = "--un";
		$option = "--un-gz" if $unaligned =~ /\.gz$/;
		$unalignedArgument .= qq{$option "$unaligned"};
	}
	my $unalignedPair = $unaligned =~ s/\.fq/_%.fq/r;
	if (@firstMates && @secondMates) {
		my $option = " --un-conc";
		$option = " --un-conc-gz" if $unalignedPair =~ /\.gz$/;
		$unalignedArgument .= qq{$option "$unalignedPair"};
	}
	$unalignedArgument .= " " if $unalignedArgument;
}

chdir "$workingDirectory";

# bowtie now ships with Python wrappers which will compile and contaminate the bundle; therefore, prevent byte compilation
$ENV{PYTHONDONTWRITEBYTECODE} = "1";

# produce the index bowtie files
print STDERR qq{"$path/bowtie2-build" --threads $threads -f "$referenceArgument" $basename\n} if $verbose;
print STDOUT qx{"$path/bowtie2-build" --threads $threads -f "$referenceArgument" $basename} if !$test;

# analyze read sequences with bowtie and convert to bam
print STDERR qq{"$path/bowtie2" --threads $threads $arguments -x $basename $unalignedArgument$matesArgument$interleavedArgument$sequenceArgument | "$path/samtools" view -bS -F 4 -o $basename.bam -\n} if $verbose;
print STDOUT qx{"$path/bowtie2" --threads $threads $arguments -x $basename $unalignedArgument$matesArgument$interleavedArgument$sequenceArgument | "$path/samtools" view -bS -F 4 -o $basename.bam -} if !$test;

# sort and index bam
print STDERR qq{"$path/samtools" sort $basename.bam -o $basename-sorted.bam\n} if $verbose;
print STDOUT qx{"$path/samtools" sort $basename.bam -o $basename-sorted.bam} if !$test;
print STDERR qq{"$path/samtools" index$indexArguments $basename-sorted.bam\n} if $verbose;
print STDOUT qx{"$path/samtools" index$indexArguments $basename-sorted.bam} if !$test;

# generate the consensus sequences
foreach my $reference (@references) {
	my ($name) = $reference =~ m{.*/(.*)\.fna};

	print STDERR qq{"$path/bcftools" mpileup --max-depth $depth -f "$reference" $basename-sorted.bam | "$path/bcftools" call -c - | "$path/vcfutils.pl" vcf2fq > $name-consensus.fq\n} if $verbose;
	print STDOUT qx{"$path/bcftools" mpileup --max-depth $depth -f "$reference" $basename-sorted.bam | "$path/bcftools" call -c - | "$path/vcfutils.pl" vcf2fq > $name-consensus.fq} if !$test;
}

sub process_signal_INT {
	kill "TERM", -$$;
	die "Process interrupted.\n";
}

__END__

print STDERR qq{\n} if $verbose;
print STDOUT qx{} if !$test;
